/* --------------------------------------------------------------
 show_logs.js 2018-08-28
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

gx.controllers.module(
    // ------------------------------------------------------------------------
    // CONTROLLER NAME
    // ------------------------------------------------------------------------
    'show_logs',

    // ------------------------------------------------------------------------
    // CONTROLLER LIBRARIES
    // ------------------------------------------------------------------------
    ['xhr'],

    // ------------------------------------------------------------------------
    // CONTROLLER BUSINESS LOGIC
    // ------------------------------------------------------------------------
    function (data) {
        'use strict';

        // ------------------------------------------------------------------------
        // VARIABLES
        // ------------------------------------------------------------------------

        /**
         * Controller reference.
         *
         * @type {jQuery}
         */
        const $this = $(this);

        /**
         * Default options for controller,
         *
         * @type {object}
         */
        const defaults = {};

        /**
         * Final controller options.
         *
         * @type {object}
         */
        const options = $.extend(true, {}, defaults, data);

        /**
         * Module object.
         *
         * @type {{}}
         */
        const module = {};

        /**
         * Logs modal html element.
         *
         * @type {jQuery|HTMLElement}
         */
        const $modal = $('.modal.logs');

        /**
         * Button that initializes the log modal on click.
         *
         * @type {jQuery|HTMLElement}
         */
        const $btnOpenLogs = $('.open-logs');

        /**
         * Body of logs modal.
         *
         * @type {jQuery|HTMLElement}
         */
        const $modalBody = $modal.find('.modal-body');

        // Event handler callback methods

        /**
         * Setup for logs modal.
         * An ajax requests the logs data from the server.
         *
         * @param event
         * @private
         */
        const _setupLogsModal = event => {
            const task = $(event.target).closest('.cronjob-element').attr('data-task');

            event.preventDefault();
            $modal.find('.modal-title').text(task);

            jse.libs.xhr.get({
                url: 'admin.php?do=CronjobAjax/getLogs&task=' + task
            }, true).done(_showLogsModal).fail(_displayError).always(() => $modal.modal('show'));
        };

        /**
         * Tear down for logs modal.
         * Will reset the modal size and title.
         *
         * @private
         */
        const _tearDownLogsModal = () => {
            $modal.find('.modal-dialog').removeClass('modal-sm').addClass('modal-lg');
            $modal.find('.modal-title').text('');
        };

        // Ajax request callback methods

        /**
         * Shows the logs modal and appends the log data to the modal body.
         *
         * @param {object} response Response of ajax that should fetch log files.
         * @param {boolean} response.success Success flag must be true if this method is called.
         * @param {string} response.log Log data.
         * @private
         */
        const _showLogsModal = response => {
            if (response.success) {
                $modalBody.empty().append(_renderLogBody(response.log));
                return;
            }

            _displayError(response);
        };

        /**
         * Displays an error message if the ajax to fetch logs failed.
         *
         * @param {object} response Response of ajax that should fetch log files.
         * @param {boolean} response.success Success flag must be false if this method is called.
         * @param {string} response.error Error message of ajax request.
         * @private
         */
        const _displayError = response => {
            const $msg = $('<p/>', {
                'text': response.error
            })

            $modal.find('.modal-dialog').removeClass('modal-lg').addClass('modal-sm');
            $modalBody.empty().append($msg);
        };

        // Private helper methods

        /**
         * Renders the modal body.
         *
         * @param {string} logData Logs data from server response.
         *
         * @returns {jQuery|HTMLElement}
         * @private
         */
        const _renderLogBody = logData => {
            const $form = $('<form/>');
            const $label = $('<label/>')
                .attr('for', 'log-messages')
                .text(jse.core.lang.translate('log_modal_last_messages', 'cronjobs'));
            const $textArea = $('<textarea/>').attr('id', 'log-messages').val(logData).prop('readonly', true);

            return $form.append($label).append($textArea);
        };

        // Module initialization
        module.init = done => {
            $btnOpenLogs.on('click', _setupLogsModal);
            $modal.on('hidden.bs.modal', _tearDownLogsModal);

            done();
        }

        return module;
    }
);